<?php

/**
 * StatusPage Admin Menu.
 *
 * Handles the main admin menu and subpages.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Admin
 *
 * Handles the plugin admin menu structure.
 */
class StatusPage_Admin
{

    /**
     * Menu slug.
     *
     * @var string
     */
    const MENU_SLUG = 'status-page-widget';

    /**
     * Initialize the admin.
     */
    public static function init()
    {
        add_action('admin_menu', array(__CLASS__, 'add_admin_menu'));
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
    }

    /**
     * Enqueue admin assets.
     *
     * @param string $hook Current admin page.
     */
    public static function enqueue_admin_assets($hook)
    {
        // Check if we're on any of our admin pages
        if (strpos($hook, self::MENU_SLUG) === false) {
            return;
        }

        wp_enqueue_style(
            'statuspage-admin',
            STATUSPAGE_WIDGET_URL . 'assets/css/admin.css',
            array(),
            STATUSPAGE_WIDGET_VERSION
        );

        // Enqueue scripts for examples page
        if (strpos($hook, 'examples') !== false) {
            wp_enqueue_script(
                'statuspage-embed',
                'https://statuspage.me/static/embed/v1/status.min.js',
                array(),
                null,
                true
            );
        }
    }

    /**
     * Add the admin menu.
     */
    public static function add_admin_menu()
    {
        // Main menu
        add_menu_page(
            __('StatusPage Widget', 'status-page-widget'),
            __('StatusPage', 'status-page-widget'),
            'manage_options',
            self::MENU_SLUG,
            array(__CLASS__, 'render_settings_page'),
            'dashicons-yes-alt',
            80
        );

        // Settings submenu (same as parent)
        add_submenu_page(
            self::MENU_SLUG,
            __('Settings', 'status-page-widget'),
            __('Settings', 'status-page-widget'),
            'manage_options',
            self::MENU_SLUG,
            array(__CLASS__, 'render_settings_page')
        );

        // Examples submenu
        add_submenu_page(
            self::MENU_SLUG,
            __('Examples', 'status-page-widget'),
            __('Examples', 'status-page-widget'),
            'manage_options',
            self::MENU_SLUG . '-examples',
            array(__CLASS__, 'render_examples_page')
        );

        // Help submenu
        add_submenu_page(
            self::MENU_SLUG,
            __('Help', 'status-page-widget'),
            __('Help', 'status-page-widget'),
            'manage_options',
            self::MENU_SLUG . '-help',
            array(__CLASS__, 'render_help_page')
        );
    }

    /**
     * Register settings.
     */
    public static function register_settings()
    {
        register_setting(
            'statuspage_widget_settings',
            StatusPage_Settings::OPTION_NAME,
            array(
                'type'              => 'array',
                'sanitize_callback' => array('StatusPage_Settings', 'sanitize_settings'),
                'default'           => StatusPage_Settings::get_default_settings(),
            )
        );

        add_settings_section(
            'statuspage_widget_defaults',
            __('Default Widget Settings', 'status-page-widget'),
            array(__CLASS__, 'render_section_description'),
            self::MENU_SLUG
        );

        // Slug field
        add_settings_field(
            'slug',
            __('Default Status Page Slug', 'status-page-widget'),
            array(__CLASS__, 'render_slug_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Theme field
        add_settings_field(
            'theme',
            __('Default Theme', 'status-page-widget'),
            array(__CLASS__, 'render_theme_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Render mode field
        add_settings_field(
            'render_mode',
            __('Default Render Mode', 'status-page-widget'),
            array(__CLASS__, 'render_mode_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Compact mode field
        add_settings_field(
            'compact',
            __('Compact Mode', 'status-page-widget'),
            array(__CLASS__, 'render_compact_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Show incidents field
        add_settings_field(
            'show_incidents',
            __('Show Incidents', 'status-page-widget'),
            array(__CLASS__, 'render_show_incidents_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Disable link field
        add_settings_field(
            'disable_link',
            __('Disable Link', 'status-page-widget'),
            array(__CLASS__, 'render_disable_link_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // RTM field
        add_settings_field(
            'rtm',
            __('Response Time (minutes)', 'status-page-widget'),
            array(__CLASS__, 'render_rtm_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // UPD field
        add_settings_field(
            'upd',
            __('Uptime (days)', 'status-page-widget'),
            array(__CLASS__, 'render_upd_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );

        // Refresh field
        add_settings_field(
            'refresh',
            __('Auto Refresh (seconds)', 'status-page-widget'),
            array(__CLASS__, 'render_refresh_field'),
            self::MENU_SLUG,
            'statuspage_widget_defaults'
        );
    }

    /**
     * Render section description.
     */
    public static function render_section_description()
    {
        echo '<p>' . esc_html__('Configure the default settings for new StatusPage widgets. These defaults will be applied when you insert a new block, widget, or shortcode.', 'status-page-widget') . '</p>';
    }

    /**
     * Render slug field.
     */
    public static function render_slug_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['slug']) ? $options['slug'] : '';
?>
        <input type="text"
            id="statuspage_slug"
            name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[slug]"
            value="<?php echo esc_attr($value); ?>"
            class="regular-text"
            pattern="^[a-z0-9]+(-[a-z0-9]+)*$"
            placeholder="your-status-page" />
        <span id="statuspage-slug-preview" class="statuspage-slug-preview"></span>
        <p class="description">
            <?php esc_html_e('Your StatusPage.me slug. Only lowercase letters (a-z), numbers (0-9), and single hyphens between characters are allowed.', 'status-page-widget'); ?>
        </p>
        <p class="description">
            <?php esc_html_e('Example: "my-company" → https://my-company.statuspage.me', 'status-page-widget'); ?>
        </p>
        <script>
            (function() {
                var input = document.getElementById('statuspage_slug');
                var preview = document.getElementById('statuspage-slug-preview');

                function validateSlug(slug) {
                    if (!slug) return {
                        valid: true,
                        message: ''
                    };
                    var pattern = /^[a-z0-9]+(-[a-z0-9]+)*$/;
                    return {
                        valid: pattern.test(slug),
                        message: slug
                    };
                }

                function updatePreview() {
                    var result = validateSlug(input.value);
                    if (!input.value) {
                        preview.textContent = '';
                        preview.className = 'statuspage-slug-preview';
                    } else if (result.valid) {
                        preview.textContent = 'https://' + input.value + '.statuspage.me';
                        preview.className = 'statuspage-slug-preview valid';
                    } else {
                        preview.textContent = '<?php esc_html_e('Invalid slug format', 'status-page-widget'); ?>';
                        preview.className = 'statuspage-slug-preview invalid';
                    }
                }

                input.addEventListener('input', updatePreview);
                updatePreview();
            })();
        </script>
    <?php
    }

    /**
     * Render theme field.
     */
    public static function render_theme_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['theme']) ? $options['theme'] : 'auto';
    ?>
        <select id="statuspage_theme" name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[theme]">
            <option value="auto" <?php selected($value, 'auto'); ?>><?php esc_html_e('Auto (System)', 'status-page-widget'); ?></option>
            <option value="light" <?php selected($value, 'light'); ?>><?php esc_html_e('Light', 'status-page-widget'); ?></option>
            <option value="dark" <?php selected($value, 'dark'); ?>><?php esc_html_e('Dark', 'status-page-widget'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('Auto will follow the user\'s system preference.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render mode field.
     */
    public static function render_mode_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['render_mode']) ? $options['render_mode'] : 'js';
    ?>
        <select id="statuspage_render_mode" name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[render_mode]">
            <option value="js" <?php selected($value, 'js'); ?>><?php esc_html_e('JavaScript (recommended)', 'status-page-widget'); ?></option>
            <option value="php" <?php selected($value, 'php'); ?>><?php esc_html_e('PHP (server-side)', 'status-page-widget'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('JavaScript mode supports auto-refresh. PHP mode is better for SEO.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render compact field.
     */
    public static function render_compact_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['compact']) ? $options['compact'] : false;
    ?>
        <label>
            <input type="checkbox"
                name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[compact]"
                value="1"
                <?php checked($value, true); ?> />
            <?php esc_html_e('Enable compact mode by default', 'status-page-widget'); ?>
        </label>
        <p class="description"><?php esc_html_e('Display a smaller, inline version of the widget.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render show incidents field.
     */
    public static function render_show_incidents_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['show_incidents']) ? $options['show_incidents'] : false;
    ?>
        <label>
            <input type="checkbox"
                name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[show_incidents]"
                value="1"
                <?php checked($value, true); ?> />
            <?php esc_html_e('Show active incidents by default', 'status-page-widget'); ?>
        </label>
    <?php
    }

    /**
     * Render disable link field.
     */
    public static function render_disable_link_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['disable_link']) ? $options['disable_link'] : false;
    ?>
        <label>
            <input type="checkbox"
                name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[disable_link]"
                value="1"
                <?php checked($value, true); ?> />
            <?php esc_html_e('Disable the link to status page by default', 'status-page-widget'); ?>
        </label>
    <?php
    }

    /**
     * Render RTM field.
     */
    public static function render_rtm_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['rtm']) ? $options['rtm'] : 60;
    ?>
        <input type="number"
            id="statuspage_rtm"
            name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[rtm]"
            value="<?php echo esc_attr($value); ?>"
            min="0"
            max="240"
            class="small-text" />
        <p class="description"><?php esc_html_e('Number of minutes of response time history to show (0-240). Set to 0 to hide.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render UPD field.
     */
    public static function render_upd_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['upd']) ? $options['upd'] : 7;
    ?>
        <input type="number"
            id="statuspage_upd"
            name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[upd]"
            value="<?php echo esc_attr($value); ?>"
            min="0"
            max="30"
            class="small-text" />
        <p class="description"><?php esc_html_e('Number of days of uptime history to show (0-30). Set to 0 to hide.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render refresh field.
     */
    public static function render_refresh_field()
    {
        $options = StatusPage_Settings::get_defaults();
        $value = isset($options['refresh']) ? $options['refresh'] : 0;
    ?>
        <input type="number"
            id="statuspage_refresh"
            name="<?php echo esc_attr(StatusPage_Settings::OPTION_NAME); ?>[refresh]"
            value="<?php echo esc_attr($value); ?>"
            min="0"
            max="300"
            step="10"
            class="small-text" />
        <p class="description"><?php esc_html_e('Auto-refresh interval in seconds (0-300). Set to 0 to disable. Only works in JavaScript mode.', 'status-page-widget'); ?></p>
    <?php
    }

    /**
     * Render settings page.
     */
    public static function render_settings_page()
    {
        if (! current_user_can('manage_options')) {
            return;
        }
    ?>
        <div class="wrap statuspage-admin-wrap">
            <h1><?php esc_html_e('StatusPage Widget Settings', 'status-page-widget'); ?></h1>

            <form action="options.php" method="post">
                <?php
                settings_fields('statuspage_widget_settings');
                do_settings_sections(self::MENU_SLUG);
                submit_button(__('Save Settings', 'status-page-widget'));
                ?>
            </form>
        </div>
    <?php
    }

    /**
     * Render examples page.
     */
    public static function render_examples_page()
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $defaults = StatusPage_Settings::get_defaults();
        $slug = !empty($defaults['slug']) ? $defaults['slug'] : 'demo';
    ?>
        <div class="wrap statuspage-admin-wrap">
            <h1><?php esc_html_e('Widget Examples', 'status-page-widget'); ?></h1>

            <p class="description">
                <?php
                if (!empty($defaults['slug'])) {
                    printf(
                        /* translators: %s: slug name */
                        esc_html__('Live examples using your configured slug: %s', 'status-page-widget'),
                        '<code>' . esc_html($slug) . '</code>'
                    );
                } else {
                    esc_html_e('Configure your default slug in Settings to see live previews. Using "demo" for examples.', 'status-page-widget');
                }
                ?>
            </p>

            <div class="statuspage-examples-grid">
                <!-- 1. Full Widget -->
                <div class="statuspage-example-card">
                    <h3><?php esc_html_e('Full Widget', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"
                            data-theme="auto"
                            data-show-incidents="1"
                            data-compact="0"
                            data-refresh="60000"
                            data-rtm="60"
                            data-upd="7"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" show_incidents="1" rtm="60" upd="7" refresh="60"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Full card with incidents, response time and uptime sparklines, auto-refresh.', 'status-page-widget'); ?></p>
                </div>

                <!-- 2. Compact Badge -->
                <div class="statuspage-example-card">
                    <h3><?php esc_html_e('Compact Badge', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"
                            data-theme="light"
                            data-compact="1"
                            data-rtm="45"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" compact="1" theme="light" rtm="45"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Compact badge with inline response time sparkline (last 45 min).', 'status-page-widget'); ?></p>
                </div>

                <!-- 3. Dark Theme -->
                <div class="statuspage-example-card statuspage-example-dark">
                    <h3><?php esc_html_e('Dark Theme', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"
                            data-theme="dark"
                            data-upd="14"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" theme="dark" upd="14"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Dark card with 14-day uptime sparkline.', 'status-page-widget'); ?></p>
                </div>

                <!-- 4. Minimal -->
                <div class="statuspage-example-card">
                    <h3><?php esc_html_e('Minimal', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" rtm="0" upd="0"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Minimal: status headline only, no sparklines.', 'status-page-widget'); ?></p>
                </div>

                <!-- 5. Compact with Incidents -->
                <div class="statuspage-example-card">
                    <h3><?php esc_html_e('Compact + Incidents', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"
                            data-theme="auto"
                            data-compact="1"
                            data-show-incidents="1"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" compact="1" show_incidents="1"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Compact with incidents list, adapts to system theme.', 'status-page-widget'); ?></p>
                </div>

                <!-- 6. Disabled Link -->
                <div class="statuspage-example-card">
                    <h3><?php esc_html_e('Static (No Link)', 'status-page-widget'); ?></h3>
                    <div class="statuspage-example-preview">
                        <script src="https://statuspage.me/static/embed/v1/status.min.js"
                            data-slug="<?php echo esc_attr($slug); ?>"
                            data-theme="auto"
                            data-show-incidents="1"
                            data-rtm="30"
                            data-upd="7"
                            data-disable-link="1"></script>
                    </div>
                    <div class="statuspage-example-code">
                        <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <code>[statuspage_widget slug="<?php echo esc_attr($slug); ?>" disable_link="1" show_incidents="1" rtm="30" upd="7"]</code>
                    </div>
                    <p class="description"><?php esc_html_e('Full widget with link disabled (static card).', 'status-page-widget'); ?></p>
                </div>
            </div>

            <hr />

            <h2><?php esc_html_e('Badge Embed', 'status-page-widget'); ?></h2>
            <p class="description"><?php esc_html_e('Use the badge shortcode for a simple status indicator image.', 'status-page-widget'); ?></p>

            <div class="statuspage-example-card" style="max-width: 400px;">
                <h3><?php esc_html_e('SVG Badge', 'status-page-widget'); ?></h3>
                <div class="statuspage-example-preview">
                    <img src="https://statuspage.me/badge/<?php echo esc_attr($slug); ?>.svg" alt="Status Badge" />
                </div>
                <div class="statuspage-example-code">
                    <h4><?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                    <code>[statuspage_badge slug="<?php echo esc_attr($slug); ?>"]</code>
                </div>
            </div>
        </div>
    <?php
    }

    /**
     * Render help page.
     */
    public static function render_help_page()
    {
        if (! current_user_can('manage_options')) {
            return;
        }
    ?>
        <div class="wrap statuspage-admin-wrap">
            <h1><?php esc_html_e('StatusPage Widget Help', 'status-page-widget'); ?></h1>

            <div class="statuspage-help-content">
                <h2><?php esc_html_e('Getting Started', 'status-page-widget'); ?></h2>
                <p><?php esc_html_e('The StatusPage Widget plugin allows you to display real-time status information from StatusPage.me on your WordPress site.', 'status-page-widget'); ?></p>

                <h3><?php esc_html_e('Step 1: Get Your Slug', 'status-page-widget'); ?></h3>
                <p>
                    <?php esc_html_e('Your status page slug is the unique identifier for your status page. It\'s the part of the URL after the domain.', 'status-page-widget'); ?>
                    <br />
                    <?php esc_html_e('For example, if your status page URL is', 'status-page-widget'); ?> <code>https://my-company.statuspage.me</code>, <?php esc_html_e('your slug is', 'status-page-widget'); ?> <code>my-company</code>.
                </p>

                <h3><?php esc_html_e('Step 2: Configure Default Settings', 'status-page-widget'); ?></h3>
                <p>
                    <?php
                    printf(
                        /* translators: %s: Settings link */
                        esc_html__('Visit the %s page to configure your default slug and other widget options.', 'status-page-widget'),
                        '<a href="' . esc_url(admin_url('admin.php?page=' . self::MENU_SLUG)) . '">' . esc_html__('Settings', 'status-page-widget') . '</a>'
                    );
                    ?>
                </p>

                <h3><?php esc_html_e('Step 3: Add the Widget', 'status-page-widget'); ?></h3>
                <p><?php esc_html_e('You can add the status widget to your site in several ways:', 'status-page-widget'); ?></p>

                <div class="statuspage-help-methods">
                    <div class="statuspage-help-method">
                        <h4><span class="dashicons dashicons-block-default"></span> <?php esc_html_e('Gutenberg Block', 'status-page-widget'); ?></h4>
                        <p><?php esc_html_e('In the block editor, search for "StatusPage" and add the StatusPage Widget block. Configure the settings in the block sidebar.', 'status-page-widget'); ?></p>
                    </div>

                    <div class="statuspage-help-method">
                        <h4><span class="dashicons dashicons-screenoptions"></span> <?php esc_html_e('Classic Widget', 'status-page-widget'); ?></h4>
                        <p><?php esc_html_e('Go to Appearance → Widgets and add the "StatusPage Widget" to any widget area.', 'status-page-widget'); ?></p>
                    </div>

                    <div class="statuspage-help-method">
                        <h4><span class="dashicons dashicons-shortcode"></span> <?php esc_html_e('Shortcode', 'status-page-widget'); ?></h4>
                        <p><?php esc_html_e('Use the shortcode in any post, page, or text widget:', 'status-page-widget'); ?></p>
                        <code>[statuspage_widget slug="your-slug"]</code>
                    </div>

                    <div class="statuspage-help-method">
                        <h4><span class="dashicons dashicons-tag"></span> <?php esc_html_e('Badge Shortcode', 'status-page-widget'); ?></h4>
                        <p><?php esc_html_e('For a simple badge image:', 'status-page-widget'); ?></p>
                        <code>[statuspage_badge slug="your-slug"]</code>
                    </div>
                </div>

                <hr />

                <h2><?php esc_html_e('Shortcode Attributes', 'status-page-widget'); ?></h2>
                <table class="widefat">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Attribute', 'status-page-widget'); ?></th>
                            <th><?php esc_html_e('Description', 'status-page-widget'); ?></th>
                            <th><?php esc_html_e('Default', 'status-page-widget'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>slug</code></td>
                            <td><?php esc_html_e('Your StatusPage.me slug', 'status-page-widget'); ?></td>
                            <td><?php esc_html_e('From settings', 'status-page-widget'); ?></td>
                        </tr>
                        <tr>
                            <td><code>theme</code></td>
                            <td><?php esc_html_e('Widget theme: auto, light, or dark', 'status-page-widget'); ?></td>
                            <td><code>auto</code></td>
                        </tr>
                        <tr>
                            <td><code>compact</code></td>
                            <td><?php esc_html_e('Compact mode: 0 or 1', 'status-page-widget'); ?></td>
                            <td><code>0</code></td>
                        </tr>
                        <tr>
                            <td><code>show_incidents</code></td>
                            <td><?php esc_html_e('Show active incidents: 0 or 1', 'status-page-widget'); ?></td>
                            <td><code>0</code></td>
                        </tr>
                        <tr>
                            <td><code>rtm</code></td>
                            <td><?php esc_html_e('Response time sparkline minutes (0-240)', 'status-page-widget'); ?></td>
                            <td><code>60</code></td>
                        </tr>
                        <tr>
                            <td><code>upd</code></td>
                            <td><?php esc_html_e('Uptime sparkline days (0-30)', 'status-page-widget'); ?></td>
                            <td><code>7</code></td>
                        </tr>
                        <tr>
                            <td><code>refresh</code></td>
                            <td><?php esc_html_e('Auto-refresh interval in seconds (0-300)', 'status-page-widget'); ?></td>
                            <td><code>0</code></td>
                        </tr>
                        <tr>
                            <td><code>disable_link</code></td>
                            <td><?php esc_html_e('Disable link to status page: 0 or 1', 'status-page-widget'); ?></td>
                            <td><code>0</code></td>
                        </tr>
                        <tr>
                            <td><code>render_mode</code></td>
                            <td><?php esc_html_e('Render mode: js or php', 'status-page-widget'); ?></td>
                            <td><code>js</code></td>
                        </tr>
                    </tbody>
                </table>

                <hr />

                <h2><?php esc_html_e('Need Help?', 'status-page-widget'); ?></h2>
                <p>
                    <?php esc_html_e('For more information, visit', 'status-page-widget'); ?>
                    <a href="https://statuspage.me" target="_blank" rel="noopener noreferrer">StatusPage.me</a>
                </p>
            </div>
        </div>
<?php
    }
}
